//
//  TTTGameViewController.m
//  Tic-Tac-Toe-Storyboard
//
//  Created by Rajiv Ramnath on 9/2/13.
//  Copyright (c) 2013 Rajiv Ramnath. All rights reserved.
//

#import "TTTGameSessionViewController.h"
#import "TTTBoard.h"
#import "TTTGameView.h"
#import "TTTGame.h"
#import "TTTSquare.h"

#import "TTTCheckpointManager.h"

@implementation TTTGameSessionViewController

    @synthesize boardView;
    @synthesize scoreTextField;
    @synthesize turnTextField;
    @synthesize changeOpponentSwitch;

    @synthesize playComputer;

    - (id)initWithNibName:(NSString *)nibNameOrNil
                   bundle:(NSBundle *)nibBundleOrNil
    {
        self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
        if (self) {
            // Custom initialization
        }
        return self;
    }

    - (void) playNewGame{
        activeGame = [[TTTGame alloc] init];
        [activeGame setPlayerNames:firstPlayerName :secondPlayerName];
        
        TTTGameGrid *gameGrid = [activeGame getGameGrid];
        [boardView enableInput];
        [boardView setGrid:gameGrid];
        
        gameView = [[TTTGameView alloc] init];
        [gameView setGameViewComponents :boardView :turnTextField :scoreTextField];
        [gameView redraw];

        [gameView showScores :[activeGame getPlayerOneName] :scorePlayerOne
                             :[activeGame getPlayerTwoName] :scorePlayerTwo];
        NSString *statusMessage =
             [[NSArray arrayWithObjects :[activeGame getCurrentPlayerName],
                                         @"to play",
                                         nil]
                                    componentsJoinedByString:@" "];
        [gameView setGameStatus :statusMessage];
    }

    - (void) accumulateScores: (NSString *) winningPlayerName{
        if(winningPlayerName==firstPlayerName)
            scorePlayerOne++;
        else
            scorePlayerTwo++;
    }


    - (void) proceedToFinish {

         NSString *winningPlayerName;
         NSString *alertMessage=nil;
        
         [boardView disableInput];
         if([activeGame isWon]) {
             winningPlayerName = [activeGame getWinningPlayerName];
             alertMessage = [[NSArray arrayWithObjects :winningPlayerName,
                                                        @"Wins!",
                                                        nil]
                              componentsJoinedByString:@" "];
             [gameView setGameStatus:alertMessage];
             [self accumulateScores:winningPlayerName];
             [gameView showScores :firstPlayerName :scorePlayerOne :secondPlayerName :scorePlayerTwo];
        } else if([activeGame isDrawn]) {
            alertMessage = @"Draw!";
            [gameView setGameStatus:alertMessage];
         }
         [gameView promptToProceed:self];
    }

    - (void)alertView:(UIAlertView *)alertView clickedButtonAtIndex:(NSInteger)buttonIndex{
        if (buttonIndex == 0){
            [self playNewGame];
        } else if (buttonIndex == 1) {
            return;
        }
    }

    - (void) humanTakesATurn :(int) x :(int) y {// human's turn
        BOOL successfulPlay = [activeGame play :x :y]; // update the model
        if (successfulPlay){
            [gameView placeSymbol :x :y]; // Refresh the display
            if([activeGame isActive]){
                NSString *statusMessage = [[NSArray arrayWithObjects :[activeGame getCurrentPlayerName],
                                                                       @"to play",
                                                                       nil]
                                            componentsJoinedByString:@" "];
                [gameView setGameStatus:statusMessage];
                if (playComputer) [self scheduleComputersTurn];
            } else {
                [self proceedToFinish];
            }
        } // else do nothing and return
    }

    - (void) computerTakesATurn {
        static int pickedX=-1, pickedY=2;

        // Select a random square to play in
        TTTGameGrid *gameGrid = [activeGame getGameGrid];
		NSArray *emptyBlocks = [gameGrid getEmptySquares];
		int n = [emptyBlocks count];
        NSLog(@"Number of empty squares = %d", n);
        int randomIndex = arc4random()%n;
        NSLog(@"Square picked = %d", randomIndex);
        TTTSquare *picked = [emptyBlocks objectAtIndex:randomIndex];
        
        // Play at that square
        pickedX = [picked x];
        pickedY = [picked y];

        [activeGame play:pickedX :pickedY];
		[gameView placeSymbol:pickedX :pickedY];
		if([activeGame isActive]) {
            [boardView enableInput];
            NSString *statusMessage = [[NSArray arrayWithObjects :[activeGame getCurrentPlayerName],
                                        @"to play",
                                        nil]
                                       componentsJoinedByString:@" "];
            [gameView setGameStatus:statusMessage];

		} else {
			[self proceedToFinish];
		}
    }

    - (void) scheduleComputersTurn {
        int randomDelay = arc4random()%TTTDELAY;
        [boardView disableInput];
        NSLog(@" Delay = >%d< seconds", randomDelay);
        [self performSelector:@selector(computerTakesATurn) withObject:nil afterDelay:randomDelay];
        // sleep(randomDelay);
        // [self computerTakesATurn];
    }


    - (void) boardTouchedAt:(float)x :(float)y {
        int posX=0, posY=0;
        NSLog(@"Touch points X %f Y %f", x, y);
        NSLog(@"Board heigth %f Width %f", [boardView getWidth], [boardView getHeight]);
        if( x > [boardView getWidth] && x < [boardView getWidth] * 2 ) posX = 1;
        if( x > [boardView getWidth] * 2 && x < [boardView getWidth] * 3 ) posX = 2;
        if( y > [boardView getHeight] && y < [boardView getHeight] * 2 ) posY = 1;
        if( y > [boardView getHeight] * 2 && y < [boardView getHeight] * 3 ) posY = 2;
        NSLog(@"X %d Y %d", posX, posY);
        [self humanTakesATurn:posX :posY];
    }

    - (void) initializeGameSession{
        NSUserDefaults *tictactoeDefaults = [NSUserDefaults standardUserDefaults];
        firstPlayerName = [tictactoeDefaults stringForKey:@"PlayerName"];
        if (firstPlayerName == nil) firstPlayerName = @"Player 1";
        
        playComputer = [tictactoeDefaults boolForKey:@"PlayComputer"];
        [changeOpponentSwitch setOn:playComputer animated:YES];
        
        secondPlayerName = @"Player 2";
        if (playComputer == YES) secondPlayerName = @"Computer";
        scorePlayerOne=0;
        scorePlayerTwo=0;
        scoreTextField.enabled = NO;
        turnTextField.enabled=NO;
        [boardView setGameSession:self];
    }

    - (void)viewDidLoad{
        [super viewDidLoad];
        // Do any additional setup after loading the view.

        TTTCheckpointManager *checkpointManager = [TTTCheckpointManager CreateManager];
        [checkpointManager registerObjectWithKey :@"TTTSessionViewController" :self];
        [self initializeGameSession];
        [self playNewGame];
    }

    - (IBAction) changeOpponent:(id)sender{
        playComputer = changeOpponentSwitch.on;
        if (![activeGame isActive])return;
        if(playComputer) {
            secondPlayerName = @"Computer";
            [self scheduleComputersTurn];
        }else{
            secondPlayerName = @"Player 2";
        }
    }

    - (void) checkpoint{
        [self synchronizeDefaults];
    }

    - (void) synchronizeDefaults {
        NSUserDefaults *tictactoeDefaults = [NSUserDefaults standardUserDefaults];
        [tictactoeDefaults setObject:[NSNumber numberWithBool:playComputer] forKey:@"PlayComputer"];
        [tictactoeDefaults synchronize];
    }

    - (IBAction) saveGame:(id)sender {
        NSDictionary* savedGameDictionary = [activeGame toDictionary];
        NSError *error;
        NSData *jsonData = [NSJSONSerialization dataWithJSONObject:savedGameDictionary
                                                           options:NSJSONWritingPrettyPrinted
                                                             error:&error];
        NSString *savedGameString = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
        
        NSString *savedGamesFilePath = [NSHomeDirectory() stringByAppendingPathComponent:@TTTGAMESESSIONSAVEDFILEPATH];

        [savedGameString writeToFile:savedGamesFilePath
                                     atomically:YES
                                     encoding:NSUTF8StringEncoding
                                     error:NULL];
    }

    - (IBAction) restoreGame:(id)sender {
        NSString *savedGamesFilePath =
            [NSHomeDirectory() stringByAppendingPathComponent:@TTTGAMESESSIONSAVEDFILEPATH];
        NSString *savedGameString = [NSString stringWithContentsOfFile:savedGamesFilePath
                                              encoding:NSUTF8StringEncoding
                                              error:NULL];
        
        NSError *restoreError  = nil;
        NSMutableDictionary *savedDictionary =
            [NSJSONSerialization JSONObjectWithData:[savedGameString dataUsingEncoding:NSUTF8StringEncoding]
                                options:NSJSONReadingMutableContainers error:&restoreError ];
        
        activeGame = [[TTTGame alloc] initFromDictionary:savedDictionary];
        TTTGameGrid *gameGrid = [activeGame getGameGrid];
        if (![activeGame isActive])[boardView disableInput];
        [boardView setGrid:gameGrid];
        [gameView redraw];
    }

    - (IBAction) exit:(id)sender {
        TTTCheckpointManager *checkpointManager = [TTTCheckpointManager CreateManager];
        [checkpointManager unregisterObjectWithKey :@"TTTSessionViewController"];
        [self synchronizeDefaults];
        [self dismissViewControllerAnimated:YES completion:nil];
    }

#pragma Telephony, Email and SMS functionality

    - (IBAction) phoneAFriend:(id)sender {
        UIApplication *applicationObject = [UIApplication sharedApplication];
        NSURL *callURL = [NSURL URLWithString:@"telprompt://614-555-1234"];
        if ([applicationObject canOpenURL:callURL]){
            [[UIApplication sharedApplication] openURL:callURL];
        } else {
            NSLog(@"Phone call failed. Probably no phone on device");
        }
    }

    - (IBAction) sendScoresBySMS:(id)sender {
        MFMessageComposeViewController *smsController = [[MFMessageComposeViewController alloc] init];
        if([MFMessageComposeViewController canSendText]){
            NSString *scoreText = [[NSArray arrayWithObjects :@"I rock at Tic-Tac-Toe! ",
                                    firstPlayerName,
                                    @":",
                                    [NSNumber numberWithInt:scorePlayerOne],
                                    @"....",
                                    secondPlayerName,
                                    @":",
                                    [NSNumber numberWithInt:scorePlayerTwo],
                                    nil]
                                   componentsJoinedByString:@" "];

            smsController.body = scoreText;
            smsController.messageComposeDelegate = self;
            [self presentViewController:smsController animated:YES completion:nil];
        }
    }

    - (void)messageComposeViewController:(MFMessageComposeViewController *)smsController
                     didFinishWithResult:(MessageComposeResult)result{
        switch (result){
        case MessageComposeResultCancelled:
            NSLog(@"SMS Message Cancelled");
            break;
        case MessageComposeResultFailed:
            NSLog(@"SMS Message Failed");
            break;
        case MessageComposeResultSent:
            NSLog(@"SMS Message Sent!");
            break;
        default:
            break;
        }
        [smsController dismissViewControllerAnimated:YES completion:nil];
    }

    - (IBAction) sendScoresByEmailWithContact:(id)sender{
        ABPeoplePickerNavigationController *picker =
        [[ABPeoplePickerNavigationController alloc] init];
        picker.peoplePickerDelegate = self;
        [self presentViewController:picker animated:YES completion:nil];
    }

    - (void)mailComposeController:(MFMailComposeViewController*)mailController
          didFinishWithResult:(MFMailComposeResult)result
                        error:(NSError*)error;
    {
        if (result == MFMailComposeResultSent) {
            NSLog(@"Email sent!");
        }
        [mailController dismissViewControllerAnimated:YES completion:nil];
    }

    - (void) sendScoresByEmailWithAddress:(NSString *)address {
        MFMailComposeViewController* mailController  = [[MFMailComposeViewController alloc] init];
        mailController.mailComposeDelegate = self;
        [mailController  setToRecipients:[NSArray arrayWithObjects :address,nil]];
        [mailController  setSubject:@"I rock at Tic-Tac-Toe!"];
    
        NSString *scoreText = [[NSArray arrayWithObjects :firstPlayerName,
                                @":",
                                [NSNumber numberWithInt:scorePlayerOne],
                                @"....",
                                secondPlayerName,
                                @":",
                                [NSNumber numberWithInt:scorePlayerTwo],
                                nil]
                               componentsJoinedByString:@" "];
    
        
        [mailController  setMessageBody:scoreText isHTML:NO];
        [self presentViewController:mailController  animated:YES completion:nil];
    }

#pragma address Book callbacks

    - (void)peoplePickerNavigationControllerDidCancel:
                (ABPeoplePickerNavigationController *)peoplePicker {
        [self dismissViewControllerAnimated:YES completion:nil];
    }


    - (BOOL)peoplePickerNavigationController:
                                (ABPeoplePickerNavigationController *)peoplePicker
                                shouldContinueAfterSelectingPerson:(ABRecordRef)selectedPerson {
        NSString* oneEmail=nil;
        ABMultiValueRef emails = ABRecordCopyValue(selectedPerson, kABPersonEmailProperty);
        if (ABMultiValueGetCount(emails) > 0) {
            oneEmail = (__bridge_transfer NSString*)
            ABMultiValueCopyValueAtIndex(emails, 0);
        }
        CFRelease(emails);
        [self dismissViewControllerAnimated:YES
                                 completion:^{[self sendScoresByEmailWithAddress:oneEmail];}];
        return NO;
    }

- (BOOL)peoplePickerNavigationController:
                                (ABPeoplePickerNavigationController *)peoplePicker
                                shouldContinueAfterSelectingPerson:(ABRecordRef)person
                                property:(ABPropertyID)property
                                identifier:(ABMultiValueIdentifier)identifier

{
    
    return NO;
    
}

#pragma Lifecycle methods

    - (void)didReceiveMemoryWarning{
        [super didReceiveMemoryWarning];
        // Dispose of any resources that can be recreated.
        NSLog(@"Game Session View Controller Memory Warning");
        [self synchronizeDefaults];
    }

@end
